<?php
/*
Plugin Name: SparkStats
Plugin URI: http://seanmcb.com/projects/wordpress/sparkstats
Description: Generates a Sparkline bar graph of your site posts and comments statistics.
Version: 0.3.1
Author: Sean McBride
Author URI: http://seanmcb.com
*/

/*  Copyright 2005  Sean McBride  (email : sean@seanmcb.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Set initial options
$sparkStats_initialOptions = array(
    'bgcolor'=>'#FFFFFF',
	'maxcolor'=>'#000000',
	'mincolor'=>'#CCCCCC',
	'maxcolortoday'=>'#990000',
	'mincolortoday'=>'#FF9999',
	'intervaltype'=>'day',
	'numintervals'=>30,
	'barwidth'=>4,
	'barspacing'=>1,
	'height'=>14,
	'renderinterval'=>60
	);
add_option('sparkStats_options',$sparkStats_initialOptions,'SparkStats Plugin Options');

// Retrieve current options
$sparkStats_options = get_option('sparkStats_options');
$sparkStats_imgDirPath = ABSPATH.'wp-content/plugins/sparkstats/curstats.png';
$sparkStats_imgWebPath = get_bloginfo('wpurl').'/wp-content/plugins/sparkstats/curstats.png';
$sparkStats_legendWebPath = get_bloginfo('wpurl').'/wp-content/plugins/sparkstats/statslegend.gif';

// Check that all options exist
$sparkStats_missingOption = false;
foreach($sparkStats_initialOptions as $k=>$v) {
    if(!isset($sparkStats_options[$k])) {
        $sparkStats_options[$k] = $v;
        $sparkStats_missingOption = true;
    }
}
if(sparkStats_missingOption) {
    update_option('sparkStats_options', $sparkStats_options);
}


// THEME FUNCTIONS

function sparkStats_imgURI() {
	echo sparkStats_get_imgURI();
}

function sparkStats_get_imgURI() {
	global $sparkStats_imgWebPath;
	return $sparkStats_imgWebPath;
}

function sparkStats_legendURI() {
	echo sparkStats_get_legendURI();
}

function sparkStats_get_legendURI() {
	global $sparkStats_legendWebPath;
	return $sparkStats_legendWebPath;
}

function sparkStats_numDays() {
    echo sparkStats_get_numDays();
}

function sparkStats_get_numDays() {
    global $sparkStats_options;
    if ($sparkStats_options['intervaltype']=='day') {
        return sparkStats_get_numIntervals();
    }
    return false;
}

function sparkStats_numIntervals() {
	echo sparkStats_get_numIntervals();
}

function sparkStats_get_numIntervals() {
	global $sparkStats_options;
	return $sparkStats_options['numintervals'];
}

function sparkStats_intervalType() {
    echo sparkStats_get_intervalType();
}

function sparkStats_get_intervalType() {
    global $sparkStats_options;
	return ucwords($sparkStats_options['intervaltype'].'s');
}

// PRIVATE PLUGIN FUNCTIONS

function sparkStats_checkAndGenerate() {
	// Check how old the statgraphic is and regenerate if it has been more than 1 hour
	global $sparkStats_imgDirPath, $sparkStats_options;
	if (file_exists($sparkStats_imgDirPath)) {
		if (time() - filemtime($sparkStats_imgDirPath) >= 60*$sparkStats_options['renderinterval']) { sparkStats_generateSparkline(); }
	} else { sparkStats_generateSparkline(); }
}

function sparkStats_generateSparkline () {
	global $wpdb, $sparkStats_options, $sparkStats_imgDirPath;
	require_once('lib/Sparkline_Bar.php');
	
	// Arrays for holding number of posts/comments per day
	$posts = array(0);
	$comments = array(0);
	
	$sql_interval = sparkStats_sql_query_interval_vals($sparkStats_options['numintervals'], $sparkStats_options['intervaltype']);
	
	// Get # of posts per day in an array
	$posts_res = $wpdb->get_results("SELECT post_date_gmt FROM $wpdb->posts WHERE DATE_SUB(CURDATE(),INTERVAL {$sql_interval[0]} {$sql_interval[1]}) <= post_date AND CURDATE() >= post_date AND post_status = 'publish' ORDER BY post_date DESC");
	$cur_day = 0;
	if ($posts_res) {
    	foreach ($posts_res as $p) {
    		while (!sparkStats_mysql_gmt_is_interval($p->post_date_gmt, $cur_day, $sparkStats_options['intervaltype'])) {
    			$cur_day++;
    			$posts[] = 0;
    			if(count($posts)>=$sparkStats_options['numintervals']) { break 2; }
    		}
    		$posts[count($posts)-1]++;
    	}
    }
	while(count($posts)<$sparkStats_options['numintervals']) { $posts[] = 0; }
	$posts = array_reverse($posts);
	// Get # of comments per day in an array
	$comments_res = $wpdb->get_results("SELECT comment_date_gmt FROM $wpdb->comments WHERE DATE_SUB(CURDATE(),INTERVAL {$sql_interval[0]} {$sql_interval[1]}) <= comment_date AND comment_approved = '1' ORDER BY comment_date DESC");
	$cur_day = 0;
	if ($comments_res) {
    	foreach ($comments_res as $c) {
    		while (!sparkStats_mysql_gmt_is_interval($c->comment_date_gmt, $cur_day, $sparkStats_options['intervaltype'])) {
    			$cur_day++;
    			$comments[] = 0;
    			if(count($comments)>=$sparkStats_options['numintervals']) { break 2; }
    		}
    		$comments[count($comments)-1]++;
    	}
    }
	while(count($comments)<$sparkStats_options['numintervals']) { $comments[] = 0; }
	$comments = array_reverse($comments);
	
	/* TEST CODE 
	echo 'Comments: ';
	foreach ($comments as $c) {
		echo $c.' ';
	}
	echo 'Posts: ';
	foreach ($posts as $c) {
		echo $c.' ';
	}
	*/
	
	// Generate new Sparkline image
	$sparkline = new Sparkline_Bar();
	$sparkline->SetDebugLevel(DEBUG_NONE);
	
	$sparkline->SetBarWidth($sparkStats_options['barwidth']);
	$sparkline->SetBarSpacing($sparkStats_options['barspacing']);
	
	// Set background color
	$color = sparkStats_check_color($sparkStats_options['bgcolor']);
	$sparkline->SetColorHtml($color, $color);
	$sparkline->SetColorHandle($color, $sparkline->DrawColorAllocate($color));
	$sparkline->SetColorBackground($color);
	
	// Draw bars
	for ($i = 0; $i < count($posts); $i++) {
		$color = sparkStats_check_color(sparkStats_getRGBForWeight($comments[$i]/max($comments), $sparkStats_options['maxcolor'], $sparkStats_options['mincolor']));
		if ($i >= count($posts)-1) { $color = sparkStats_getRGBForWeight($comments[$i]/max($comments), $sparkStats_options['maxcolortoday'], $sparkStats_options['mincolortoday']); }
		$sparkline->SetColorHtml($color, $color);
		$sparkline->SetColorHandle($color, $sparkline->DrawColorAllocate($color));
		$sparkline->SetData($i, $posts[$i], $color);
	}
	
	// Render Sparkline image
	$sparkline->Render($sparkStats_options['height']);
	
	// Save Sparkline image at image path
	$sparkline->Output($sparkStats_imgDirPath);

}

function sparkStats_getRGBForWeight ($weight, $maxrgb = '#000000', $minrgb = '#EEEEEE') {
	// Adapted from UltimateTagWarrior plugin

	$minr = hexdec(substr($minrgb, 1, 2));
	$ming = hexdec(substr($minrgb, 3, 2));
	$minb = hexdec(substr($minrgb, 5, 2));

	$maxr = hexdec(substr($maxrgb, 1, 2));
	$maxg = hexdec(substr($maxrgb, 3, 2));
	$maxb = hexdec(substr($maxrgb, 5, 2));

	$r = dechex(intval((($maxr - $minr) * $weight) + $minr));
	$g = dechex(intval((($maxg - $ming) * $weight) + $ming));
	$b = dechex(intval((($maxb - $minb) * $weight) + $minb));

	if (strlen($r) == 1) $r = "0" . $r;
	if (strlen($g) == 1) $g = "0" . $g;
	if (strlen($b) == 1) $b = "0" . $b;

	return strtoupper("#$r$g$b");
}

function sparkStats_wp_addOptionsPage() {
	if (function_exists('add_options_page')) {
		add_options_page('SparkStats', 'SparkStats', 8, basename(__FILE__), 'sparkStats_wp_optionsPage');
	}
}

function sparkStats_wp_optionsPage() {
	global $sparkStats_options;
	if (isset($_POST['info_update'])) { ?>
		<div id="message" class="updated fade"><p><strong>
		<?php 
		// Verify and update submitted info
		if(!(preg_match('%^#[ABCDEFabcdef\d]{6}$%',$_POST['sparkStats_maxcolor']) && 
				preg_match('%^#[ABCDEFabcdef\d]{6}$%',$_POST['sparkStats_mincolor']) &&
				preg_match('%^#[ABCDEFabcdef\d]{6}$%',$_POST['sparkStats_bgcolor']) &&
				preg_match('%^#[ABCDEFabcdef\d]{6}$%',$_POST['sparkStats_maxcolortoday']) &&
				preg_match('%^#[ABCDEFabcdef\d]{6}$%',$_POST['sparkStats_mincolortoday']))) {
			echo 'Colors options must be in valid RGB format. (Ex: #996627) Try again.';
		} elseif (!(preg_match('%^\d+$%',$_POST['sparkStats_barwidth']) &&
				preg_match('%^\d+$%',$_POST['sparkStats_barwidth']) &&
				preg_match('%^\d+$%',$_POST['sparkStats_barwidth']))) {
			echo 'Size options must be positive integers. Try again.';
		} elseif (!(preg_match('%^\d+$%',$_POST['sparkStats_numintervals']) &&
		        preg_match('%^\d+$%',$_POST['sparkStats_renderinterval']))) {
			echo 'Number of days and rendering interval must be a positive integers. Try again.';
	    } elseif (!in_array($_POST['sparkStats_intervaltype'],array('day','week','month'))) {
    	    echo 'Invalid interval type. Please choose again.';
		} else {
			// Options are valid, save them
			foreach($sparkStats_options as $k=>$v) {
				$sparkStats_options[$k] = $_POST['sparkStats_'.$k];
			}
			update_option('sparkStats_options', $sparkStats_options);
			sparkStats_generateSparkline();
			echo 'SparkStats options have been updated successfully.';
		}
		?>
		</strong></p></div>
	<?php } ?>
	<div class=wrap>
		<form method="post">
			<h2>SparkStats Options</h2>
			<p>Current graph image: <em>(you may have to refresh to see changes)</em></p>
			<p><img src="<?php sparkStats_imgURI(); ?>" alt="SparkStats Graph" style="padding:5px; background-color: <?php echo $sparkStats_options['bgcolor']; ?>;" /></p>
			<p>The height of each bar represents the number of posts during that interval. The color of each line represents the number of comments during that interval.</p>
			<p>Updating your SparkStats options will also regenerate your current SparkStats image.</p>
			<p><fieldset name="set_color">
				<legend><strong><?php _e('Color Options', 'Localization name') ?></strong></legend>
				<ul>
				    <li><label for="sparkStats_bgcolor">Background Color:</label> <input name="sparkStats_bgcolor" type="text" maxlength"7" value="<?php echo $sparkStats_options['bgcolor']; ?>"></li>
					<li><label for="sparkStats_maxcolor">Maximum Value Color:</label> <input name="sparkStats_maxcolor" type="text" maxlength="7" value="<?php echo $sparkStats_options['maxcolor']; ?>"></li>
					<li><label for="sparkStats_mincolor">Minimum Value Color:</label> <input name="sparkStats_mincolor" type="text" maxlength="7" value="<?php echo $sparkStats_options['mincolor']; ?>"></li>
					<li><label for="sparkStats_maxcolortoday">Maximum Value Color (for Today):</label> <input name="sparkStats_maxcolortoday" type="text" maxlength="7" value="<?php echo $sparkStats_options['maxcolortoday']; ?>"></li>
					<li><label for="sparkStats_mincolortoday">Minimum Value Color (for Today):</label> <input name="sparkStats_mincolortoday" type="text" maxlength="7" value="<?php echo $sparkStats_options['mincolortoday']; ?>"></li>
				</ul>
			</fieldset></p>
			<p><fieldset name="set_size">
				<legend><strong><?php _e('Size Options', 'Localization name') ?></strong></legend>
				<ul>
					<li><label for="sparkStats_barwidth">Bar Width (px):</label> <input name="sparkStats_barwidth" type="text" value="<?php echo $sparkStats_options['barwidth']; ?>"></li>
					<li><label for="sparkStats_barspacing">Spacing Between Bars (px):</label> <input name="sparkStats_barspacing" type="text" value="<?php echo $sparkStats_options['barspacing']; ?>"></li>
					<li><label for="sparkStats_height">Graph Height (px):</label> <input name="sparkStats_height" type="text" value="<?php echo $sparkStats_options['height']; ?>"></li>
				</ul>
			</fieldset></p>
			<p><fieldset name="set_time">
				<legend><strong><?php _e('Time Options', 'Localization name') ?></strong></legend>
				<ul>
				    <li>
				        <label for="sparkStats_intervaltype">Bars Represent:</label>
				        <label><input name="sparkStats_intervaltype" id="sparkStats_intervaltype_day" type="radio" value="day" <?php if($sparkStats_options['intervaltype']=='day'){ echo 'checked="true"'; } ?> /> Days</label>
				        <label><input name="sparkStats_intervaltype" id="sparkStats_intervaltype_week" type="radio" value="week" <?php if($sparkStats_options['intervaltype']=='week'){ echo 'checked="true"'; } ?> /> Weeks</label>
				        <label><input name="sparkStats_intervaltype" id="sparkStats_intervaltype_month" type="radio" value="month" <?php if($sparkStats_options['intervaltype']=='month'){ echo 'checked="true"'; } ?> /> Months</label>
				        <?php if ($sparkStats_options['intervaltype']=='week') { ?>
				        <br/><em>(When weeks is selected, the "Weeks in the calendar should start on" option in the "General" options tab will affect the SparkStats output.)</em>
				        <?php } ?>
				    </li>
					<li><label for="sparkStats_numintervals">Number of Intervals to Show:</label> <input name="sparkStats_numintervals" type="text" value="<?php echo $sparkStats_options['numintervals']; ?>"></li>
				</ul>
			</fieldset></p>
			<p><fieldset name="set_render">
				<legend><strong><?php _e('Render Options', 'Localization name') ?></strong></legend>
				<ul>
					<li><label for="sparkStats_renderinterval">Remake graph every X minutes:</label> <input name="sparkStats_renderinterval" type="text" value="<?php echo $sparkStats_options['renderinterval']; ?>"></li>
				</ul>
			</fieldset></p>
			<p><em>Powered by the <a href="http://sparkline.org" title="Sparkline PHP Graphing Library">Sparkline PHP Graphing Library</a>. Plugin by <a href="http://www.seanmcb.com" title="Sean McBride">Sean McBride</a> of <a href="http://www.alwaysbeta.com" title="alwaysBETA">alwaysBETA</a>.</em></p>
			<div class="submit"><input type="submit" name="info_update" value="<?php _e('Update options', 'Localization name') ?>" /></div>
		</form>
	</div>
<?php
}

function sparkStats_sql_query_interval_vals($intervals, $interval_type) {
    switch ($interval_type) {
    case 'day':
        return array($intervals, 'DAY');
    case 'week':
        $gmt_offset = get_option('gmt_offset');
        $week_start = get_option('start_of_week');
        $now_gmt_offset = time()+($gmt_offset*60*60);
        $days = ($intervals-1)*7+((gmdate('w',$now_gmt_offset)-$week_start)%7)+1;
        return array($days, 'DAY');
    case 'month':
        return array($intervals, 'MONTH');
    }
}

function sparkStats_mysql_gmt_is_interval($mysql_timestamp, $interval_offset, $interval_type) {
    switch($interval_type) {
    case 'day':
        return sparkStats_mysql_gmt_is_day($mysql_timestamp, $interval_offset);
    case 'week':
        return sparkStats_mysql_gmt_is_week($mysql_timestamp, $interval_offset);
    case 'month':
        return sparkStats_mysql_gmt_is_month($mysql_timestamp, $interval_offset);
    }
    return false;
}

function sparkStats_mysql_gmt_is_day($mysql_timestamp, $day_offset){
	$gmt_offset = get_option('gmt_offset');
	preg_match('/(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/', $mysql_timestamp,$pieces);
	$mysql_gmt_offset = gmmktime($pieces[4],$pieces[5],$pieces[6],$pieces[2],$pieces[3],$pieces[1])+($gmt_offset*60*60)+($day_offset*24*60*60);
	$now_gmt_offset = time()+($gmt_offset*60*60);
	return (gmdate('Y',$mysql_gmt_offset)==gmdate('Y',$now_gmt_offset) && gmdate('m',$mysql_gmt_offset)==gmdate('m',$now_gmt_offset) && gmdate('d',$mysql_gmt_offset)==gmdate('d',$now_gmt_offset));
}

function sparkStats_mysql_gmt_is_week($mysql_timestamp, $week_offset){
	$gmt_offset = get_option('gmt_offset');
	preg_match('/(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/', $mysql_timestamp,$pieces);
	$mysql_gmt_offset = gmmktime($pieces[4],$pieces[5],$pieces[6],$pieces[2],$pieces[3],$pieces[1])+($gmt_offset*60*60)+($week_offset*7*24*60*60);
	$now_gmt_offset = time()+($gmt_offset*60*60);
	$week_start = get_option('start_of_week');
	$week_start_gmt_offset = gmmktime(0,0,0,gmdate('n',$now_gmt_offset),gmdate('j',$now_gmt_offset),gmdate('Y',$now_gmt_offset))+($week_start-gmdate('w',$now_gmt_offset))*24*60*60;
	return ($mysql_gmt_offset >= $week_start_gmt_offset && $mysql_gmt_offset < $week_start_gmt_offset+7*24*60*60);
}

function sparkStats_mysql_gmt_is_month($mysql_timestamp, $month_offset){
	$gmt_offset = get_option('gmt_offset');
	preg_match('/(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/', $mysql_timestamp,$pieces);
	$mysql_gmt_offset = gmmktime($pieces[4],$pieces[5],$pieces[6],$pieces[2]+($month_offset%12),$pieces[3],$pieces[1]+floor($month_offset/12))+($gmt_offset*60*60);
	$now_gmt_offset = time()+($gmt_offset*60*60);
	return (gmdate('Y',$mysql_gmt_offset)==gmdate('Y',$now_gmt_offset) && gmdate('m',$mysql_gmt_offset)==gmdate('m',$now_gmt_offset));
}

function sparkStats_check_color($color) {
    if (strtolower($color)=='#000000') { return '#010101'; }
    return $color;
}

// Add hooks and filters
add_action('admin_menu', 'sparkStats_wp_addOptionsPage');
add_action('shutdown', 'sparkStats_checkAndGenerate');

?>
